﻿using System;
using System.Collections.Generic;
using System.DirectoryServices;
using System.Linq;
using System.Text;
using EleWise.ELMA.Runtime.Settings;
using EleWise.ELMA.Model.Attributes;
using EleWise.ELMA.Security.Cryptography;

namespace EleWise.ELMA.IntegrationLdap
{

    /// <summary>
    /// Настройки для модуля интеграция с LDAP
    /// </summary>
    public class IntegrationLdapSettings : GlobalSettingsBase
    {

        /// <summary>
        /// Определяет, что пароль не изменен
        /// </summary>
        public const string PASSWORD_NOT_CHANGED = "($PASSWORD_NOT_CHANGED$)";

        /// <summary>
        /// Ctor
        /// </summary>
        public IntegrationLdapSettings()
        {
            LdapUrl = string.Empty;
            LdapAuthType = (int)AuthenticationTypes.None;
            LdapPath = string.Empty;
            LdapParamLogin = string.Empty;
            LdapParamName = string.Empty;
            LdapParamSecond = string.Empty;
            LdapParamMiddle = string.Empty;
            LdapEMail = string.Empty;
            LdapLogin = string.Empty;
            //LdapPassword = string.Empty;
            LdapAuthLogin = string.Empty;
            LdapAuthFilter = string.Empty;
        }

        /// <summary>
        /// Адрес сервера
        /// </summary>
        [DisplayName(typeof(IntegrationLdapSettings_SR), "LdapUrl")]
        public string LdapUrl { get; set; }

        /// <summary>
        /// Тип авторизации
        /// </summary>
        [DisplayName(typeof(IntegrationLdapSettings_SR), "LdapAuthType")]
        public int LdapAuthType { get; set; }

        /// <summary>
        /// Путь к пользователям
        /// </summary>
        [DisplayName(typeof(IntegrationLdapSettings_SR), "LdapPath")]
        public string LdapPath { get; set; }

        /// <summary>
        /// Параметр "Логин"
        /// </summary>
        [DisplayName(typeof(IntegrationLdapSettings_SR), "LdapParamLogin")]
        public string LdapParamLogin { get; set; }

        /// <summary>
        /// Параметр "Имя"
        /// </summary>
        [DisplayName(typeof(IntegrationLdapSettings_SR), "LdapParamName")]
        public string LdapParamName { get; set; }

        /// <summary>
        /// Параметр "Фамилия"
        /// </summary>
        [DisplayName(typeof(IntegrationLdapSettings_SR), "LdapParamSecond")]
        public string LdapParamSecond { get; set; }

        /// <summary>
        /// Параметр "Отчество"
        /// </summary>
        [DisplayName(typeof(IntegrationLdapSettings_SR), "LdapParamMiddle")]
        public string LdapParamMiddle { get; set; }

        /// <summary>
        /// Параметр "Email"
        /// </summary>
        [DisplayName(typeof(IntegrationLdapSettings_SR), "LdapEMail")]
        public string LdapEMail { get; set; }

        /// <summary>
        /// Пользователь
        /// </summary>
        [DisplayName(typeof(IntegrationLdapSettings_SR), "LdapLogin")]
        public string LdapLogin { get; set; }

        /// <summary>
        /// Пароль
        /// </summary>
        [DisplayName(typeof(IntegrationLdapSettings_SR), "LdapPassword")]
        public string LdapPassword
        {
            get { return ldapPassword; }
            set
            {
                if (value != PASSWORD_NOT_CHANGED)
                {
                    ldapPassword = value;
                }
            }
        }

        /// <summary>
        /// Фильтр для авторизации
        /// </summary>
        [DisplayName(typeof(IntegrationLdapSettings_SR), "LdapAuthLogin")]
        public string LdapAuthLogin { get; set; }

        /// <summary>
        /// Дополнительный фильтр для импорта
        /// </summary>
        [DisplayName(typeof(IntegrationLdapSettings_SR), "LdapAuthFilter")]
        public string LdapAuthFilter { get; set; }


        public static Dictionary<int, string> GetUsedLdapAuthType()
        {
            var result = new Dictionary<int, string>();
            result.Add((int)AuthenticationTypes.None,GetLdapAuthTypeName((int)AuthenticationTypes.None));
            result.Add((int)AuthenticationTypes.Encryption,GetLdapAuthTypeName((int)AuthenticationTypes.Encryption));
            result.Add((int)AuthenticationTypes.Secure,GetLdapAuthTypeName((int)AuthenticationTypes.Secure));
            result.Add((int)AuthenticationTypes.ReadonlyServer,GetLdapAuthTypeName((int)AuthenticationTypes.ReadonlyServer));

            return result;
        }

        /// <summary>
        /// Получить тип авторизации
        /// </summary>
        /// <param name="dmsVersionSetting"></param>
        /// <returns></returns>
        public static string GetLdapAuthTypeName(int ldapAuthType)
        {
            switch (ldapAuthType)
            {
                case (int)AuthenticationTypes.None:
                    return SR.T("Базовая");
                    break;
                case (int)AuthenticationTypes.Encryption:
                    return SR.T("С шифрованием");
                    break;
                case (int)AuthenticationTypes.Secure:
                    return SR.T("Безопасная");
                    break;
                case (int)AuthenticationTypes.ReadonlyServer:
                    return SR.T("Только для чтения");
                    break;
            }
            return string.Empty;
        }



        #region Protected Members

        protected override IDictionary<string, string> GetStorageValue(System.Reflection.PropertyInfo property)
        {
            var v = base.GetStorageValue(property);
            switch (property.Name)
            {
                case "LdapPassword":
                    return new Dictionary<string, string> { { property.Name, ElmaCryptoProvider.EncryptPassword(LdapPassword) } };
                default:
                    return v;
            }
        }

        protected override void SetStorageValue(System.Reflection.PropertyInfo property, IDictionary<string, string> values)
        {
            switch (property.Name)
            {
                case "LdapPassword":
                    string stringValue;
                    if (values.TryGetValue(property.Name, out stringValue))
                    {
                        var value = ElmaCryptoProvider.DecryptPassword(stringValue);
                        LdapPassword = value;
                    }
                    break;
                default:
                    base.SetStorageValue(property, values);
                    break;
            }
        }

        #endregion


        #region Private Members

        private string ldapPassword;

        #endregion


        
    }


    #region Resources

    internal static class IntegrationLdapSettings_SR
    {
        public static string LdapUrl { get { return SR.T("Адрес сервера"); } }
        public static string LdapAuthType { get { return SR.T("Тип авторизации"); } }
        public static string LdapPath { get { return SR.T("Путь к пользователям"); } }
        public static string LdapParamLogin { get { return SR.T("Параметр \"Логин\""); } }
        public static string LdapParamName { get { return SR.T("Параметр \"Имя\""); } }
        public static string LdapParamSecond { get { return SR.T("Параметр \"Фамилия\""); } }
        public static string LdapParamMiddle { get { return SR.T("Параметр \"Отчество\""); } }
        public static string LdapEMail { get { return SR.T("Параметр \"Email\""); } }
        public static string LdapLogin { get { return SR.T("Пользователь"); } }
        public static string LdapPassword { get { return SR.T("Пароль"); } }
        public static string LdapAuthLogin { get { return SR.T("Фильтр для авторизации"); } }
        public static string LdapAuthFilter { get { return SR.T("Дополнительный фильтр для импорта"); } }
    }
    #endregion

}
