﻿using System;
using System.DirectoryServices;
using EleWise.ELMA.ComponentModel;
using EleWise.ELMA.Logging;
using EleWise.ELMA.Runtime.Managers;
using EleWise.ELMA.Security;
using EleWise.ELMA.Security.Services;
using EleWise.ELMA.Services;

namespace EleWise.ELMA.IntegrationLdap.Extensions
{
    [Component]
    internal class LdapUserModelMembershipEventHandler : IMembershipServiceEventHandler
    {

        /// <summary>
        /// Узел/объект LDAP. Для поиска пользователя начиная с указаного в настройках узла. Под логином-паролем из настроек.
        /// </summary>
        /// <param name="settings"></param>
        /// <returns></returns>
        private DirectoryEntry GetDirectoryEntry(IntegrationLdapSettings settings)
        {
            if (settings == null)
                throw new Exception(SR.T("Отсутствуют настройки LDAP"));

            var directoryEntry = new DirectoryEntry
            {
                Path = "LDAP://" + settings.LdapUrl + "/" + settings.LdapPath,
                AuthenticationType = (AuthenticationTypes)settings.LdapAuthType,
                Username = settings.LdapLogin,
                Password = settings.LdapPassword
            };
            return directoryEntry;
        }

        /// <summary>
        /// Узел/объект LDAP. Ранее найденый пользователь. Под логином-паролем введенными пользователем.
        /// </summary>
        /// <param name="settings"></param>
        /// <returns></returns>
        private DirectoryEntry GetDirectoryEntry(IntegrationLdapSettings settings, string path)
        {
            if (settings == null)
                throw new Exception(SR.T("Отсутствуют настройки LDAP"));

            var directoryEntry = new DirectoryEntry
            {
                Path = path,
                AuthenticationType = (AuthenticationTypes)settings.LdapAuthType,
                Username = path.Remove(0, ((string)("LDAP://" + settings.LdapUrl + "/")).Length ),
                Password = settings.LdapPassword
            };
            return directoryEntry;
        }

        /// <summary>
        /// Получение строки поиска
        /// </summary>
        /// <param name="settings"></param>
        /// <param name="searchString"></param>
        /// <returns></returns>
        protected string GetSearchString(IntegrationLdapSettings settings, string searchString)
        {
            string s = "";

            if (string.IsNullOrWhiteSpace(settings.LdapParamLogin))
                throw new ArgumentNullException("LdapLogin");
            if (string.IsNullOrWhiteSpace(searchString))
                throw new ArgumentNullException("Login");

            s = string.Format("({0}={1})", settings.LdapParamLogin, searchString);
            if (!string.IsNullOrWhiteSpace(settings.LdapAuthFilter))
                s = string.Format("(&{0}{1})", settings.LdapAuthFilter, s);

            return s;
        }

        //Валидация по уникальному имени
        protected bool ValidatingDN(UserValidationContext context)
        {
            var module = Locator.GetServiceNotNull<IntegrationLdapSettingsModule>();

            var settings = module.Settings;
            if (settings == null || string.IsNullOrWhiteSpace(settings.LdapUrl))
            {
                Logging.Logger.Log.Error(SR.T("Отсутствуют настройки LDAP"));
                return false;
            }

            //ищем всех пользователей с указаным логином в OU и глубже взятом из "Путь к пользователям"
            var directoryEntry = GetDirectoryEntry(settings);
            var directorySearcher = new DirectorySearcher(directoryEntry, GetSearchString(settings, context.User.UserName));
            SearchResultCollection searchResultCollection = directorySearcher.FindAll();

            //такой пользователь должен быть только один
            if (searchResultCollection == null)
                throw new Exception("searchResultCollection == null");
            if (searchResultCollection.Count == 0)
                throw new Exception("User not found in LDAP");
            if (searchResultCollection.Count > 1)
                throw new Exception("It is more than one user in LDAP");

            //получаем искомого пользователя
            var searchResult = searchResultCollection[0];

            var authSettings = new IntegrationLdapSettings();
            var values = DataAccessManager.SettingsManager.GetAllStrings(module.ModuleGuid);
            if (values != null)
                authSettings.SetStorageValues(values);

            if ((authSettings.LdapAuthType == (int)AuthenticationTypes.None || authSettings.LdapAuthType == (int)AuthenticationTypes.ReadonlyServer)
                && string.IsNullOrEmpty(context.Password))
                return false;

            authSettings.LdapLogin = (authSettings.LdapAuthLogin == null)
                                     ? context.User.UserName
                                     : authSettings.LdapAuthLogin.Replace("{$login}", context.User.UserName);

            authSettings.LdapPassword = context.Password;

            //пытаемся получить объект, являющийся самим пользователем под логином-паролем пользователя
            using (var lc = GetDirectoryEntry(authSettings, searchResult.Path))
            {
                var n = lc.NativeObject;
            }
            return true;
        }

        //Валидация по шаблону
        protected bool ValidatingTemplate(UserValidationContext context)
        {
            var module = Locator.GetServiceNotNull<IntegrationLdapSettingsModule>();

            var settings = module.Settings;
            if (settings == null || string.IsNullOrWhiteSpace(settings.LdapUrl))
            {
                Logging.Logger.Log.Error(SR.T("Отсутствуют настройки LDAP"));
                return false;
            }

            var authSettings = new IntegrationLdapSettings();
            var values = DataAccessManager.SettingsManager.GetAllStrings(module.ModuleGuid);
            if (values != null) authSettings.SetStorageValues(values);

            if ((authSettings.LdapAuthType == (int)AuthenticationTypes.None
                || authSettings.LdapAuthType == (int)AuthenticationTypes.ReadonlyServer)
                && string.IsNullOrEmpty(context.Password))
                return false;

            authSettings.LdapLogin = (authSettings.LdapAuthLogin == null)
                                     ? context.User.UserName
                                     : authSettings.LdapAuthLogin.Replace("{$login}", context.User.UserName);
            authSettings.LdapPassword = context.Password;
            using (var lc = GetDirectoryEntry(authSettings))
            {
                var n = lc.NativeObject;
            }

            return true;
        }

        /// <summary>
        /// Начало проверки авторизации
        /// </summary>
        /// <param name="context">Контекст проверки авторизации</param>
        public void Validating(UserValidationContext context)
        {
            var lLdapExternalMembershipService = Locator.GetServiceNotNull<LdapExternalMembershipService>();

            //Авторизация выполняется только для пользователей, привязанных к провайдеру авторизации LDAP
            if (context.User.AuthProviderGuid == lLdapExternalMembershipService.ServiceUid)
            {
                if (context.Authorized) return;

                //попытка авторизации по DN
                try
                {
                    if (!ValidatingDN(context))
                        return;
                    
                }
                catch (Exception ex1)
                {
                    //попытка авторизации по шаблону
                    try
                    {
                        if (!ValidatingTemplate(context))
                            return;    
                    }
                    catch (Exception ex2)
                    {
                        context.Authorized = false;
                        Logger.Log.Error(string.Format("LDAP DN fail. Login: {0}. Exception: {1}", context.User.UserName, ex1.Message), ex1);
                        Logger.Log.Error(string.Format("LDAP Template fail. Login: {0}. Exception: {1}", context.User.UserName, ex2.Message), ex2);
                        return;
                    }
                }

                context.Authorized = true;
            }
        }


        /// <summary>
        /// Проверка авторизации завершена (возможно и неуспешно)
        /// </summary>
        /// <param name="context">Контекст проверки авторизации</param>
        public void Validated(UserValidationContext context)
        {
        }


        /// <summary>
        /// Смена пароля пользователю
        /// </summary>
        /// <param name="user"></param>
        /// <param name="password"></param>
        public void SetPassword(IUser user, string password)
        {
        }

    }
}
