﻿using System;
using System.Collections.Generic;
using System.DirectoryServices;
using System.Linq;
using System.Text;
using EleWise.ELMA.ComponentModel;
using EleWise.ELMA.Logging;
using EleWise.ELMA.Security;
using EleWise.ELMA.Security.Managers;
using EleWise.ELMA.Services;

namespace EleWise.ELMA.IntegrationLdap.Extensions
{
    [Component]
    public class LdapExternalMembershipService : IExternalMembershipService
    {

        /// <summary>
        /// Узел/объект LDAP
        /// </summary>
        /// <param name="settings"></param>
        /// <returns></returns>
        private DirectoryEntry GetDirectoryEntry(IntegrationLdapSettings settings)
        {
            if (settings == null)
                throw new Exception(SR.T("Отсутствуют настройки LDAP"));

            var directoryEntry = new DirectoryEntry
            {
                Path = "LDAP://" + settings.LdapUrl + "/" + settings.LdapPath,
                AuthenticationType = (AuthenticationTypes)settings.LdapAuthType,
                Username = settings.LdapLogin,
                Password = settings.LdapPassword
            };
            return directoryEntry;
        }

        /// <summary>
        /// Получение строки поиска
        /// </summary>
        /// <param name="settings"></param>
        /// <param name="searchString"></param>
        /// <returns></returns>
        protected string GetSearchString(IntegrationLdapSettings settings, string searchString)
        {
            string s = "";

            if (!string.IsNullOrWhiteSpace(searchString))
            {
                if (!string.IsNullOrWhiteSpace(settings.LdapParamLogin))
                    s = string.Format("({0}=*{1}*)", settings.LdapParamLogin, searchString);
                else
                    throw new ArgumentNullException("LdapLogin");

                if (!string.IsNullOrWhiteSpace(settings.LdapParamName))
                    s = string.Format("(|({0}=*{1}*){2})", settings.LdapParamName, searchString, s);

                if (!string.IsNullOrWhiteSpace(settings.LdapParamSecond))
                    s = string.Format("(|({0}=*{1}*){2})", settings.LdapParamSecond, searchString, s);

                if (!string.IsNullOrWhiteSpace(settings.LdapEMail))
                    s = string.Format("(|({0}=*{1}*){2})", settings.LdapEMail, searchString, s);

                if (!string.IsNullOrWhiteSpace(settings.LdapParamMiddle))
                    s = string.Format("(|({0}=*{1}*){2})", settings.LdapParamMiddle, searchString, s);

                if (!string.IsNullOrWhiteSpace(settings.LdapAuthFilter))
                    s = string.Format("(&{0}{1})", settings.LdapAuthFilter, s);
            }
            else
            {
                if (!string.IsNullOrWhiteSpace(settings.LdapAuthFilter))
                    s = string.Format("{0}", settings.LdapAuthFilter);
            }

            return s;
        }

        /// <summary>
        /// Получение строки поиска для пользователя
        /// </summary>
        /// <param name="settings"></param>
        /// <param name="searchString"></param>
        /// <returns></returns>
        protected string GetUserSearchString(IntegrationLdapSettings settings, string searchString)
        {
            string s = "";

            if (!string.IsNullOrWhiteSpace(searchString))
            {
                if (!string.IsNullOrWhiteSpace(settings.LdapParamLogin))
                    s = string.Format("({0}={1})", settings.LdapParamLogin, searchString);
                else
                    throw new ArgumentNullException("LdapLogin");

                if (!string.IsNullOrWhiteSpace(settings.LdapAuthFilter))
                    s = string.Format("(&{0}{1})", settings.LdapAuthFilter, s);

            }

            return s;
        }

        /// <summary>
        /// Считывание данных о пользователе из результата поиска
        /// </summary>
        /// <param name="searchResult"></param>
        /// <param name="settings"></param>
        /// <returns></returns>
        protected EleWise.ELMA.Security.Models.IUser ReadUserParam(SearchResult searchResult, IntegrationLdapSettings settings)
        { 
            var user = UserManager.Instance.Create();

            //Пытаемся прочитать ФИО и Адрес электронной почты из LDAP
            try
            {
                user.UserName = searchResult.Properties[settings.LdapParamLogin][0].ToString();
            }
            catch (Exception exception)
            {
                var error = string.Format("Login read fail. Param: {0}", settings.LdapParamLogin);
                Logging.Logger.Log.Error(error, exception);

            }

            try
            {
                user.FirstName = searchResult.Properties[settings.LdapParamName][0].ToString();
            }
            catch
            { }

            try
            {
                user.LastName = searchResult.Properties[settings.LdapParamSecond][0].ToString();
            }
            catch
            { }

            try
            {
                user.MiddleName = searchResult.Properties[settings.LdapParamMiddle][0].ToString();
            }
            catch
            { }

            user.FullName = "";
            user.FullName = (user.LastName ?? "") + " " + (user.FirstName ?? "") + " " + (user.MiddleName ?? "");

            try
            {
                user.EMail = searchResult.Properties[settings.LdapEMail][0].ToString();
            }
            catch (Exception exception)
            {
                var error = string.Format("Email read fail. Param: {0}", settings.LdapEMail);
                Logging.Logger.Log.Error(error, exception);
            }

            return user;
        }


        #region Реализация интерфейса

        /// <summary>
        /// Уникальный идентификатор сервиса
        /// </summary>
        public Guid ServiceUid { get { return new Guid("7EFF5F19-58D1-4904-8699-0A059BAB5F08"); } }

        public string Description { get {return SR.T("LDAP"); } }

        /// <summary>
        /// Проверить авторизацию пользователя
        /// </summary>
        /// <param name="userNameOrEmail">Имя пользователя</param>
        /// <param name="password">Пароль</param>
        /// <returns><c>true</c>, если пользователь успешно авторизован</returns>
        public bool ValidateUser(string userNameOrEmail, string password)
        {
            return false;
        }


        ///<summary>
        /// Синхронизировать данные о пользователе
        ///</summary>
        ///<param name="user">Пользователь</param>
        public void Sync(IUser user)
        {
            if (user == null)
            {
                Logging.Logger.Log.Error("Synchronization Error: user is null");
                return;
            }


            var module = Locator.GetServiceNotNull<IntegrationLdapSettingsModule>();

            if (module == null)
            {
                Logging.Logger.Log.Error("Synchronization Error: IntegrationLdapSettingsModule not found");
                return;
            }

            var settings = module.Settings;

            if (settings == null)
            {
                Logging.Logger.Log.Error(SR.T("Отсутствуют настройки LDAP"));
                return;
            }

            if (string.IsNullOrWhiteSpace(settings.LdapUrl))
            {
                Logging.Logger.Log.Error(SR.T("Отсутствует адрес LDAP сервера"));
                return;
            }

            //Получение списка пользователей из LDAP по строке поиска
            var directoryEntry = GetDirectoryEntry(settings);
            
            string searchString = GetUserSearchString(settings, user.UserName);
            if (string.IsNullOrWhiteSpace(searchString))
            {
                Logging.Logger.Log.Error(string.Format("Synchronization Error: can't read from LDAP {0}", user.UserName));
                return;
            }
            var directorySearcher = new DirectorySearcher(directoryEntry, searchString);

            SearchResultCollection searchResultCollection = directorySearcher.FindAll();

            if (searchResultCollection != null && searchResultCollection.Count > 0)
            {
                var userLDAP = ReadUserParam(searchResultCollection[0], settings);
                ((EleWise.ELMA.Security.Models.IUser)user).FirstName = userLDAP.FirstName;
                ((EleWise.ELMA.Security.Models.IUser)user).LastName = userLDAP.LastName;
                ((EleWise.ELMA.Security.Models.IUser)user).MiddleName = userLDAP.MiddleName;
                ((EleWise.ELMA.Security.Models.IUser)user).EMail = userLDAP.EMail;
            }
            else
                Logger.Log.Error(SR.T("Не удалось получить дополнительную информацию о пользователе LDAP"));

        }


        ///<summary>
        /// Поиск пользователей по подстроке
        ///</summary>
        ///<param name="searchString">Строка поиска</param>
        ///<returns>Список пользователей</returns>
        public IEnumerable<IUser> FindUsers(string searchString)
        {
            var result = new List<IUser>();

            try
            {
            
                if (searchString == null) 
                    return result;

                var module = Locator.GetServiceNotNull<IntegrationLdapSettingsModule>();
                if (module == null) return result;
                var settings = module.Settings;

                if (settings == null)
                    throw new Exception(SR.T("Отсутствуют настройки LDAP"));
                if (string.IsNullOrWhiteSpace(settings.LdapUrl))
                    throw new Exception(SR.T("Отсутствует адрес LDAP сервера"));
                if (string.IsNullOrWhiteSpace(settings.LdapParamLogin))
                    throw new Exception(SR.T("Отсутствует настройка параметра \"Логин\""));


                //Получение списка пользователей из LDAP по строке поиска
                var directoryEntry = GetDirectoryEntry(settings);
                var directorySearcher = new DirectorySearcher(directoryEntry, GetSearchString(settings, searchString));
                SearchResultCollection searchResultCollection = directorySearcher.FindAll();

                //Формирование результирующего списка IUser
                foreach (SearchResult searchResult in searchResultCollection)
                { 
                    try
                    {
                        var user = ReadUserParam(searchResult, settings);
                        user.AuthProviderGuid = ServiceUid;

                        result.Add(user);
                    }
                    catch (Exception ex)
                    {
                        var error = string.Format("{0}: {1}", "Import user from LDAP failed", ex.Message);
                        Logging.Logger.Log.Error(error, ex);
                    }
                }

            }
            catch (Exception ex)
            {
                var error = string.Format("{0}: {1}", SR.T("Не удалось получить список пользователей из LDAP"), ex.Message);
                Logging.Logger.Log.Error(error, ex);
                throw new Exception(error);
            }


            return result;
        }


        /// <summary>
        /// Получить настройки сервиса
        /// </summary>
        /// <returns>Настройки сервиса</returns>
        public MembershipSettings GetSettings()
        {
            return null;
        }

        #endregion
    }
}
